using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using DataContracts;
using System.Runtime.CompilerServices;

namespace Ato.EN.IntegrationServices.CodeGenerationCHTWTHHLDTFN
{

    public partial class CHTWTHHLD2018Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="CHTWTHHLD2018Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CHTWTHHLD2018Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }
				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public CHTWTHHLD2018 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CHTWTHHLD2018 report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;


    
            #region VR.ATO.CHTWTHHLD.000003
    
            /*  VR.ATO.CHTWTHHLD.000003
            Trust Postal Suburb/Town is mandatory
    
            Legacy Rule Format:
            IF [chtwthhld5] = NULLORBLANK
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld5] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.LocalityName.Text

            Technical Business Rule Format:
            ^CHTWTHHLD5 = BLANK
    
            Data Elements:
    
            ^CHTWTHHLD5 = RP.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD5) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430245", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000003"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD5", Value = report.CHTWTHHLD5 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000003
    
            #region VR.ATO.CHTWTHHLD.000005
    
            /*  VR.ATO.CHTWTHHLD.000005
            The TFN report is a quarterly report.  The dates provided should be one of the four following periods for a year: 1 July - 30 September; 1 October - 31 December; 1 January - 31 March or 1 April - 30 June.
    
            Legacy Rule Format:
            IF (Day(RP:period.startDate) <> 1) OR (MonthAsInt(RP:period.startDate) <> 1 AND MonthAsInt(RP:period.startDate) <> 4 AND MonthAsInt(RP:period.startDate) <> 7 AND MonthAsInt(RP:period.startDate) <> 10) OR (Year(RP:period.startDate) <> Year(RP:period.endDate)) OR (MonthAsInt(RP:period.startDate) = 7 AND ((MonthAsInt(RP:period.endDate) <> 9) OR (Day(RP:period.endDate) <> 30))) OR (MonthAsInt(RP:period.startDate) = 10 AND ((MonthAsInt(RP:period.endDate) <> 12) OR (Day(RP:period.endDate) <> 31))) OR (MonthAsInt(RP:period.startDate) = 1 AND ((MonthAsInt(RP:period.endDate) <> 3) OR (Day(RP:period.endDate) <> 31))) OR (MonthAsInt(RP:period.startDate) = 4 AND ((MonthAsInt(RP:period.endDate) <> 6) OR (Day(RP:period.endDate) <> 30)))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (Day(^StartDate1) <> 1) OR (MonthAsInt(^StartDate2) <> 1 AND MonthAsInt(^StartDate3) <> 4 AND MonthAsInt(^StartDate4) <> 7 AND MonthAsInt(^StartDate5) <> 10) OR (Year(^StartDate6) <> Year(^EndDate)) OR (MonthAsInt(^StartDate7) = 7 AND ((MonthAsInt(^EndDate) <> 9) OR (Day(^EndDate) <> 30))) OR (MonthAsInt(^StartDate8) = 10 AND ((MonthAsInt(^EndDate) <> 12) OR (Day(^EndDate) <> 31))) OR (MonthAsInt(^StartDate9) = 1 AND ((MonthAsInt(^EndDate) <> 3) OR (Day(^EndDate) <> 31))) OR (MonthAsInt(^StartDate10) = 4 AND ((MonthAsInt(^EndDate) <> 6) OR (Day(^EndDate) <> 30)))
    
            Data Elements:
    
            ^EndDate = RP
    
            ^StartDate1 = RP
    
            ^StartDate2 = RP
    
            ^StartDate3 = RP
    
            ^StartDate4 = RP
    
            ^StartDate5 = RP
    
            ^StartDate6 = RP
    
            ^StartDate7 = RP
    
            ^StartDate8 = RP
    
            ^StartDate9 = RP
    
            ^StartDate10 = RP
            */
            assertion = (Day(report.RPStartDate) != 1 || MonthAsInt(report.RPStartDate) != 1 && MonthAsInt(report.RPStartDate) != 4 && MonthAsInt(report.RPStartDate) != 7 && MonthAsInt(report.RPStartDate) != 10 || Year(report.RPStartDate) != Year(report.RPEndDate) || MonthAsInt(report.RPStartDate) == 7 && (MonthAsInt(report.RPEndDate) != 9 || Day(report.RPEndDate) != 30) || MonthAsInt(report.RPStartDate) == 10 && (MonthAsInt(report.RPEndDate) != 12 || Day(report.RPEndDate) != 31) || MonthAsInt(report.RPStartDate) == 1 && (MonthAsInt(report.RPEndDate) != 3 || Day(report.RPEndDate) != 31) || MonthAsInt(report.RPStartDate) == 4 && (MonthAsInt(report.RPEndDate) != 6 || Day(report.RPEndDate) != 30));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000005", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN report period dates are not for a quarter",
                    LongDescription = @"The TFN report is a quarterly report.  The dates provided must be one of the four following periods for a year: 1 July - 30 September; 1 October - 31 December; 1 January - 31 March or 1 April - 30 June.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000005"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000005
    
            #region VR.ATO.CHTWTHHLD.000006
    
            /*  VR.ATO.CHTWTHHLD.000006
            In the 2009 budget the government announced its intention to extend the tax file number (TFN) withholding arrangements to closely held trusts, including family trusts, with effect from the 2010/11 income year. From 1 July 2010 trustees of closely held tru
    
            Legacy Rule Format:
            IF RP:period.startDate < 2010-07-01
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 < AsDate('2010-07-01')
    
            Data Elements:
    
            ^StartDate1 = RP
            */
            assertion = ((report.RPExists == true) && (report.RPStartDate < AsDate(@"2010-07-01")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000006", Severity = ProcessMessageSeverity.Error,
                    Description = @"The TFN report period must not be earlier than July 2010",
                    LongDescription = @"The TFN report Reporting period from date must not be earlier than July 2010 as the requirement for trustees to withhold tax only came into effect from 1 July 2010.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000006"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000006
    
            #region VR.ATO.CHTWTHHLD.000007
    
            /*  VR.ATO.CHTWTHHLD.000007
            Country name is present with State or postcode in the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            IF ([chtwthhld8] <> NULL) AND ([chtwthhld8] <> "au") AND (([chtwthhld6] <> NULL) OR ([chtwthhld7] <> NULL))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld6] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhld7] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.Postcode.Text
            [chtwthhld8] = CHTWTHHLD:RP.Postal:pyde.02.08:AddressDetails.Country.Code

            Technical Business Rule Format:
            (^CHTWTHHLD8 <> NULL) AND (^CHTWTHHLD8 <> 'au') AND ((^CHTWTHHLD6 <> NULL) OR (^CHTWTHHLD7 <> NULL))
    
            Data Elements:
    
            ^CHTWTHHLD8 = RP.Postal:AddressDetails.Country.Code
    
            ^CHTWTHHLD6 = RP.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLD7 = RP.Postal:AddressDetails.Postcode.Text
            */
            assertion = (report.CHTWTHHLD8 != null && report.CHTWTHHLD8 != @"au" && (report.CHTWTHHLD6 != null || report.CHTWTHHLD7 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000007", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and postcode must not be present for non-Australian addresses",
                    LongDescription = @"Country name is present with State or Postcode in the Trust postal address field.  Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000007"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD8", Value = report.CHTWTHHLD8 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD6", Value = report.CHTWTHHLD6 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD7", Value = report.CHTWTHHLD7 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000007
    
            #region VR.ATO.CHTWTHHLD.000008
    
            /*  VR.ATO.CHTWTHHLD.000008
            State is present in the address field without a postcode.
    
            Legacy Rule Format:
            IF ([chtwthhld6] <> NULL) AND ([chtwthhld7] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld6] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhld7] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            (^CHTWTHHLD6 <> NULL) AND (^CHTWTHHLD7 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLD6 = RP.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLD7 = RP.Postal:AddressDetails.Postcode.Text
            */
            assertion = (report.CHTWTHHLD6 != null && report.CHTWTHHLD7 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode must be present if State is present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD6", Value = report.CHTWTHHLD6 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD7", Value = report.CHTWTHHLD7 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000008
    
            #region VR.ATO.CHTWTHHLD.000009
    
            /*  VR.ATO.CHTWTHHLD.000009
            Postcode is present in the address field without a State code.
    
            Legacy Rule Format:
            IF ([chtwthhld7] <> NULLORBLANK) AND ([chtwthhld6] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld6] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhld7] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.Postcode.Text

            Technical Business Rule Format:
            (^CHTWTHHLD7 <> BLANK) AND (^CHTWTHHLD6 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD6 = RP.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLD7 = RP.Postal:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD7) != true && string.IsNullOrWhiteSpace(report.CHTWTHHLD6) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410017", Severity = ProcessMessageSeverity.Error,
                    Description = @"State must be present if Postcode is present",
                    LongDescription = @"If the address is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000009"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD7", Value = report.CHTWTHHLD7 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD6", Value = report.CHTWTHHLD6 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000009
    
            #region VR.ATO.CHTWTHHLD.000010
    
            /*  VR.ATO.CHTWTHHLD.000010
            State, postcode, or country are missing from the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            IF ([chtwthhld6] = NULL) AND ([chtwthhld7] = NULL) AND (([chtwthhld8] = NULL) OR ([chtwthhld8] = "au"))
                        RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld6] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.StateOrTerritory.Code
            [chtwthhld7] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.Postcode.Text
            [chtwthhld8] = CHTWTHHLD:RP.Postal:pyde.02.08:AddressDetails.Country.Code

            Technical Business Rule Format:
            (^CHTWTHHLD6 = NULL) AND (^CHTWTHHLD7 = NULL) AND ((^CHTWTHHLD8 = NULL) OR (^CHTWTHHLD8 = 'au'))
    
            Data Elements:
    
            ^CHTWTHHLD6 = RP.Postal:AddressDetails.StateOrTerritory.Code
    
            ^CHTWTHHLD7 = RP.Postal:AddressDetails.Postcode.Text
    
            ^CHTWTHHLD8 = RP.Postal:AddressDetails.Country.Code
            */
            assertion = (report.CHTWTHHLD6 == null && report.CHTWTHHLD7 == null && (report.CHTWTHHLD8 == null || report.CHTWTHHLD8 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000010", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and postcode are required for Australian addresses",
                    LongDescription = @"State, postcode, or country are required in the Trust postal address field.  Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD6", Value = report.CHTWTHHLD6 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD7", Value = report.CHTWTHHLD7 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD8", Value = report.CHTWTHHLD8 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000010
    
            #region VR.ATO.CHTWTHHLD.000014
    
            /*  VR.ATO.CHTWTHHLD.000014
            Trust Name is mandatory
    
            Legacy Rule Format:
            IF [chtwthhld2] = NULLORBLANK
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld2] = CHTWTHHLD:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            ^CHTWTHHLD2 = BLANK
    
            Data Elements:
    
            ^CHTWTHHLD2 = RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD2) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410215", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000014"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD2", Value = report.CHTWTHHLD2 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000014
    
            #region VR.ATO.CHTWTHHLD.000016
    
            /*  VR.ATO.CHTWTHHLD.000016
            Total amount withheld and total amount of payments must not be present for the quarterly report.
    
            Legacy Rule Format:
            IF ([chtwthhld10] <> NULL) OR ([chtwthhld9] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld9] = CHTWTHHLD:RP:rvctc2.02.12:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount
            [chtwthhld10] = CHTWTHHLD:RP:rvctc2.02.12:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount

            Technical Business Rule Format:
            (^CHTWTHHLD10 <> NULL) OR (^CHTWTHHLD9 <> NULL)
    
            Data Elements:
    
            ^CHTWTHHLD10 = RP:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount
    
            ^CHTWTHHLD9 = RP:IncomeTax.PayAsYouGoWithholding.PaymentGross.Amount
            */
            assertion = (report.CHTWTHHLD10 != null || report.CHTWTHHLD9 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000016", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total amount withheld and Total amount of payments must not be present",
                    LongDescription = @"Trust total amount withheld and Trust total amount of payments must not be present for the TFN report.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000016"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD10", Value = report.CHTWTHHLD10.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD9", Value = report.CHTWTHHLD9.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000016
    
            #region VR.ATO.CHTWTHHLD.000019
    
            /*  VR.ATO.CHTWTHHLD.000019
            Trustee First given name has a maximum field length of 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.GivenName.Text) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD14) > 15
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = (Length(report.CHTWTHHLD14) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000019", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee first given name must not exceed 15 characters",
                    LongDescription = @"For Given name, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000019"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000019
    
            #region VR.ATO.CHTWTHHLD.000021
    
            /*  VR.ATO.CHTWTHHLD.000021
            Trust Postal Street name and number is mandatory
    
            Legacy Rule Format:
            IF [chtwthhld3] = NULLORBLANK
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld3] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.Line1.Text

            Technical Business Rule Format:
            ^CHTWTHHLD3 = BLANK
    
            Data Elements:
    
            ^CHTWTHHLD3 = RP.Postal:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD3) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address must be supplied.",
                    LongDescription = @"Only one instance of current postal address must be supplied.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000021"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD3", Value = report.CHTWTHHLD3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000021
    
            #region VR.ATO.CHTWTHHLD.000029
    
            /*  VR.ATO.CHTWTHHLD.000029
            Trust Name has a maximum field length of 76
    
            Legacy Rule Format:
            IF LENGTH([chtwthhld2]) > 76
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld2] = CHTWTHHLD:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            Length(^CHTWTHHLD2) > 76
    
            Data Elements:
    
            ^CHTWTHHLD2 = RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (Length(report.CHTWTHHLD2) > 76);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust name must not exceed 76 characters",
                    LongDescription = @"For Trust name, the maximum allowable for this form is 76 characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000029"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD2", Value = report.CHTWTHHLD2 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000029
    
            #region VR.ATO.CHTWTHHLD.000030
    
            /*  VR.ATO.CHTWTHHLD.000030
            Trust Postal Suburb/Town has a maximum field length of 27
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.LocalityName.Text) > 27
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD5) > 27
    
            Data Elements:
    
            ^CHTWTHHLD5 = RP.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (Length(report.CHTWTHHLD5) > 27);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust postal suburb/town must not exceed 27 characters",
                    LongDescription = @"For Trust postal suburb/town, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD5", Value = report.CHTWTHHLD5 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000030
    
            #region VR.ATO.CHTWTHHLD.000032
    
            /*  VR.ATO.CHTWTHHLD.000032
            Trustee Family name has a maximum field length of 30
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.FamilyName.Text) > 30
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD12) > 30
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (Length(report.CHTWTHHLD12) > 30);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000032", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee family name must not exceed 30 characters",
                    LongDescription = @"For Trustee family name, the maximum allowable characters for this form is 30",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000032"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000032
    
            #region VR.ATO.CHTWTHHLD.000034
    
            /*  VR.ATO.CHTWTHHLD.000034
            Family name or non-individual name must be present.
    
            Legacy Rule Format:
            IF ([chtwthhld12] = NULLORBLANK) AND ([chtwthhld16] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld12] = CHTWTHHLD:Trustee:pyde.02.00:PersonNameDetails.FamilyName.Text
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CHTWTHHLD12 = BLANK) AND (^CHTWTHHLD16 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD12) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD16) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000034", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name must be present",
                    LongDescription = @"Trustee family name or Trustee non-individual name must be present.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000034"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000034
    
            #region VR.ATO.CHTWTHHLD.000035
    
            /*  VR.ATO.CHTWTHHLD.000035
            Non-individual name and Family name are both present. Only one name can be present.
    
            Legacy Rule Format:
            IF ([chtwthhld16] <> NULLORBLANK) AND ([chtwthhld12] <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld12] = CHTWTHHLD:Trustee:pyde.02.00:PersonNameDetails.FamilyName.Text
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CHTWTHHLD16 <> BLANK) AND (^CHTWTHHLD12 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD16) != true && string.IsNullOrWhiteSpace(report.CHTWTHHLD12) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Both family name and company name present",
                    LongDescription = @"Trustee family name and Trustee non-individual name are both present. Only family or non-individual name for Trustee may be present.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000035
    
            #region VR.ATO.CHTWTHHLD.000037
    
            /*  VR.ATO.CHTWTHHLD.000037
            Daytime phone number is present without daytime phone area code.
    
            Legacy Rule Format:
            IF ([chtwthhld19] <> NULLORBLANK)
            AND ([chtwthhld18] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld18] = CHTWTHHLD:Trustee:pyde.02.00:ElectronicContact.Telephone.Area.Code
            [chtwthhld19] = CHTWTHHLD:Trustee:pyde.02.00:ElectronicContact.Telephone.Minimal.Number

            Technical Business Rule Format:
            (^CHTWTHHLD19 <> BLANK) AND (^CHTWTHHLD18 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD18 = Trustee:ElectronicContact.Telephone.Area.Code
    
            ^CHTWTHHLD19 = Trustee:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD19) != true && string.IsNullOrWhiteSpace(report.CHTWTHHLD18) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000037", Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"Trustee daytime phone area code is required when Trustee daytime phone number is provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000037"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD19", Value = report.CHTWTHHLD19 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD18", Value = report.CHTWTHHLD18 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000037
    
            #region VR.ATO.CHTWTHHLD.000050
    
            /*  VR.ATO.CHTWTHHLD.000050
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            IF (([chtwthhld8] = NULL) OR ([chtwthhld8] = "au")) AND ([chtwthhld5] CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld5] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.LocalityName.Text
            [chtwthhld8] = CHTWTHHLD:RP.Postal:pyde.02.08:AddressDetails.Country.Code

            Technical Business Rule Format:
            ((^CHTWTHHLD8 = NULL) OR (^CHTWTHHLD8 = 'au')) AND (ContainsSet(^CHTWTHHLD5, '0-9'))
    
            Data Elements:
    
            ^CHTWTHHLD5 = RP.Postal:AddressDetails.LocalityName.Text
    
            ^CHTWTHHLD8 = RP.Postal:AddressDetails.Country.Code
            */
            assertion = ((report.CHTWTHHLD8 == null || report.CHTWTHHLD8 == @"au") && IsMatch(report.CHTWTHHLD5, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000050"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD8", Value = report.CHTWTHHLD8 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD5", Value = report.CHTWTHHLD5 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000050
    
            #region VR.ATO.CHTWTHHLD.000051
    
            /*  VR.ATO.CHTWTHHLD.000051
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            IF (([chtwthhld8] = NULL) OR ([chtwthhld8] = "au")) AND ([chtwthhld5] = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld5] = CHTWTHHLD:RP.Postal:pyde.02.00:AddressDetails.LocalityName.Text
            [chtwthhld8] = CHTWTHHLD:RP.Postal:pyde.02.08:AddressDetails.Country.Code

            Technical Business Rule Format:
            ((^CHTWTHHLD8 = NULL) OR (^CHTWTHHLD8 = 'au')) AND (FoundSet(^CHTWTHHLD5, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLD5 = RP.Postal:AddressDetails.LocalityName.Text
    
            ^CHTWTHHLD8 = RP.Postal:AddressDetails.Country.Code
            */
            assertion = ((report.CHTWTHHLD8 == null || report.CHTWTHHLD8 == @"au") && IsMatch(report.CHTWTHHLD5, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000051"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD8", Value = report.CHTWTHHLD8 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD5", Value = report.CHTWTHHLD5 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000051
    
            #region VR.ATO.CHTWTHHLD.000055
    
            /*  VR.ATO.CHTWTHHLD.000055
            Trustee Other given names has a maximum field length of 30
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.OtherGivenName.Text) > 30
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD15) > 30
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (Length(report.CHTWTHHLD15) > 30);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee other given names must not exceed 30 characters",
                    LongDescription = @"For Trustee other given names, the maximum allowable characters for this form is 30",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000055
    
            #region VR.ATO.CHTWTHHLD.000062
    
            /*  VR.ATO.CHTWTHHLD.000062
            Trustee Non-Individual Name has a maximum field length of 76
    
            Legacy Rule Format:
            IF LENGTH([chtwthhld16]) > 76
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            Length(^CHTWTHHLD16) > 76
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (Length(report.CHTWTHHLD16) > 76);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000062", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee non-individual name must not exceed 76 characters",
                    LongDescription = @"For Trustee non-individual name, the maximum allowable characters for this form is 76",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000062"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000062
    
            #region VR.ATO.CHTWTHHLD.000084
    
            /*  VR.ATO.CHTWTHHLD.000084
            Non-individual name field starts with 'T/A' followed by a space and ends with a leading space followed by 'Pship', 'P'ship' or 'P/ship'. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF ([chtwthhld16] STARTSWITH "T/A ") AND ([chtwthhld16] ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (StartsWith(^CHTWTHHLD16, 'T/A ')) AND (EndsWithSet(^CHTWTHHLD16, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.CHTWTHHLD16, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.CHTWTHHLD16, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000084"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000084
    
            #region VR.ATO.CHTWTHHLD.000085
    
            /*  VR.ATO.CHTWTHHLD.000085
            Non-individual name field ends with a leading space followed by 'T/A', 'T/A P'ship', 'T/A Pship', 'T/A P/Ship' or 'T/A Partnership'. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF ([chtwthhld16] ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (EndsWithSet(^CHTWTHHLD16, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD16, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000085"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000085
    
            #region VR.ATO.CHTWTHHLD.000090
    
            /*  VR.ATO.CHTWTHHLD.000090
            TFN in the context definition must pass the TFN algorithm check
    
            Legacy Rule Format:
            IF (CONTEXT(RP.TrueAndCorrect)<>NULL) AND (TFNALGORITHM(RP.TrueAndCorrect:entity.identifier.TFN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (FailsTFNAlgorithm(^TFN))
    
            Data Elements:
    
            ^Context = RP:RP.TrueAndCorrect
    
            ^TFN = RP:RP.TrueAndCorrect
            */
            assertion = (report.RPTrueAndCorrectExists != false && FailsTFNAlgorithm(report.RPTrueAndCorrectIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000090", Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party declarer Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000090"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000090
    
            #region VR.ATO.CHTWTHHLD.000093
    
            /*  VR.ATO.CHTWTHHLD.000093
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD79) > 15
    
            Data Elements:
    
            ^CHTWTHHLD79 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.CHTWTHHLD79) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000093", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone numbers must not exceed 15 characters",
                    LongDescription = @"For Telephone numbers, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000093"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD79", Value = report.CHTWTHHLD79 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000093
    
            #region VR.ATO.CHTWTHHLD.000093
    
            /*  VR.ATO.CHTWTHHLD.000093
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD87) > 15
    
            Data Elements:
    
            ^CHTWTHHLD87 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.CHTWTHHLD87) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000093", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone numbers must not exceed 15 characters",
                    LongDescription = @"For Telephone numbers, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000093"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD87", Value = report.CHTWTHHLD87 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000093
    
            #region VR.ATO.CHTWTHHLD.000093
    
            /*  VR.ATO.CHTWTHHLD.000093
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD19) > 15
    
            Data Elements:
    
            ^CHTWTHHLD19 = Trustee:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.CHTWTHHLD19) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000093", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone numbers must not exceed 15 characters",
                    LongDescription = @"For Telephone numbers, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000093"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD19", Value = report.CHTWTHHLD19 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000093
    
            #region VR.ATO.CHTWTHHLD.000100
    
            /*  VR.ATO.CHTWTHHLD.000100
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? /# or a space character.
    
            Legacy Rule Format:
            IF ([chtwthhld2] <> NULLORBLANK) AND ([chtwthhld2] CONTAINS SET("{","}","|","_","\"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld2] = CHTWTHHLD:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD2, '"{","}","|","_","\"')
    
            Data Elements:
    
            ^CHTWTHHLD2 = RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD2, @"\{|\}|\||_|\\",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000100", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD2", Value = report.CHTWTHHLD2 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000100
    
            #region VR.ATO.CHTWTHHLD.000101
    
            /*  VR.ATO.CHTWTHHLD.000101
            Trust address line 1 has a maximum field length of 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD3) > 38
    
            Data Elements:
    
            ^CHTWTHHLD3 = RP.Postal:AddressDetails.Line1.Text
            */
            assertion = (Length(report.CHTWTHHLD3) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000101", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Trust postal street name and number must not exceed 38 characters",
                    LongDescription = @"For each Trust postal street name and number, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000101"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD3", Value = report.CHTWTHHLD3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000101
    
            #region VR.ATO.CHTWTHHLD.000103
    
            /*  VR.ATO.CHTWTHHLD.000103
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? /# or a space character.
    
            Legacy Rule Format:
            IF ([chtwthhld16] <> NULLORBLANK) AND ([chtwthhld16] CONTAINS SET("{","}","|","_","\"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD16, '"{","}","|","_","\"')
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD16, @"\{|\}|\||_|\\",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000103", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee non-individual name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000103"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000103
    
            #region VR.ATO.CHTWTHHLD.000104
    
            /*  VR.ATO.CHTWTHHLD.000104
            Daytime phone area code is present without daytime phone number.
    
            Legacy Rule Format:
            IF ([chtwthhld19] = NULLORBLANK)
            AND ([chtwthhld18] <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld18] = CHTWTHHLD:Trustee:pyde.02.00:ElectronicContact.Telephone.Area.Code
            [chtwthhld19] = CHTWTHHLD:Trustee:pyde.02.00:ElectronicContact.Telephone.Minimal.Number

            Technical Business Rule Format:
            (^CHTWTHHLD19 = BLANK) AND (^CHTWTHHLD18 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD18 = Trustee:ElectronicContact.Telephone.Area.Code
    
            ^CHTWTHHLD19 = Trustee:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD19) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD18) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000104", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"Trustee daytime phone number is required when Trustee daytime phone area code is provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000104"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD19", Value = report.CHTWTHHLD19 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD18", Value = report.CHTWTHHLD18 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000104
    
            #region VR.ATO.CHTWTHHLD.000106
    
            /*  VR.ATO.CHTWTHHLD.000106
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line1.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD3 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD3, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD3 = RP.Postal:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD3) != true && !(IsMatch(report.CHTWTHHLD3, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000106", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust postal street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000106"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD3", Value = report.CHTWTHHLD3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000106
    
            #region VR.ATO.CHTWTHHLD.000107
    
            /*  VR.ATO.CHTWTHHLD.000107
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.Line2.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD4 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD4, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD4 = RP.Postal:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD4) != true && !(IsMatch(report.CHTWTHHLD4, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000107", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust postal street name and number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000107"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD4", Value = report.CHTWTHHLD4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000107
    
            #region VR.ATO.CHTWTHHLD.000108
    
            /*  VR.ATO.CHTWTHHLD.000108
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:AddressDetails.LocalityName.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD5 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD5, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD5 = RP.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD5) != true && !(IsMatch(report.CHTWTHHLD5, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000108", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust postal suburb/town contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000108"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD5", Value = report.CHTWTHHLD5 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000108
    
            #region VR.ATO.CHTWTHHLD.000110
    
            /*  VR.ATO.CHTWTHHLD.000110
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD12, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD12, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000110", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee family name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000110"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000110
    
            #region VR.ATO.CHTWTHHLD.000111
    
            /*  VR.ATO.CHTWTHHLD.000111
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD14, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD14, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000111", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee first given name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000111"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000111
    
            #region VR.ATO.CHTWTHHLD.000114
    
            /*  VR.ATO.CHTWTHHLD.000114
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD15, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD15, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000114", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee other given names contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000114"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000114
    
            #region VR.ATO.CHTWTHHLD.000115
    
            /*  VR.ATO.CHTWTHHLD.000115
            Trust address line 2 has a maximum field length of 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD4) > 38
    
            Data Elements:
    
            ^CHTWTHHLD4 = RP.Postal:AddressDetails.Line2.Text
            */
            assertion = (Length(report.CHTWTHHLD4) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000115", Severity = ProcessMessageSeverity.Error,
                    Description = @"Each Trust postal street name and number must not exceed 38 characters",
                    LongDescription = @"For each Trust postal street name and number, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000115"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD4", Value = report.CHTWTHHLD4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000115
    
            #region VR.ATO.CHTWTHHLD.000125
    
            /*  VR.ATO.CHTWTHHLD.000125
            The TFN has failed the algorithm check.
    
            Legacy Rule Format:
            IF (RP:entity.identifier.TFN <> NULLORBLANK) AND (TFNALGORITHM (RP:entity.identifier.TFN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (FailsTFNAlgorithm(^TFN))
    
            Data Elements:
    
            ^TFN = RP
            */
            assertion = (string.IsNullOrWhiteSpace(report.RPIdentifierTFN) != true && FailsTFNAlgorithm(report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000125", Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000125"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000125
    
            #region VR.ATO.CHTWTHHLD.000126
    
            /*  VR.ATO.CHTWTHHLD.000126
            The TFN of the reporting party should be the same as the TFN of the postal reporting party
    
            Legacy Rule Format:
            IF (RP.Postal:entity.identifier.TFN <> NULLORBLANK) AND (RP.Postal:entity.identifier.TFN <> RP:entity.identifier.TFN)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (^TFN <> ^OtherTFN)
    
            Data Elements:
    
            ^TFN = RP:RP.Postal
    
            ^OtherTFN = RP
            */
            assertion = ((report.RPPostalExists == true) && (string.IsNullOrWhiteSpace(report.RPPostalIdentifierTFN) != true && report.RPPostalIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000126", Severity = ProcessMessageSeverity.Error,
                    Description = @"Check postal reporting party TFN",
                    LongDescription = @"The Postal Reporting Party tax file number should be the same as the Reporting Party TFN.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPostalOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000126"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000126
    
            #region VR.ATO.CHTWTHHLD.000127
    
            /*  VR.ATO.CHTWTHHLD.000127
            Scheme for RP.Postal must be set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF (RP.Postal:entity.identifier.scheme <> “http://www.ato.gov.au/tfn”)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Postal
            */
            assertion = ((report.RPPostalExists == true) && (report.RPPostalIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000127", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty.Postal must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPostalOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000127"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000127
    
            #region VR.ATO.CHTWTHHLD.000130
    
            /*  VR.ATO.CHTWTHHLD.000130
            Non-individual name field contains 'P/L' which is not a valid abbreviation. This abbreviation cannot be used.
    
            Legacy Rule Format:
            IF ([chtwthhld16] CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [chtwthhld16] = CHTWTHHLD:Trustee:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            Contains(^CHTWTHHLD16, 'P/L')
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CHTWTHHLD16 == null ? false : report.CHTWTHHLD16.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000130"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000130
    
            #region VR.ATO.CHTWTHHLD.000131
    
            /*  VR.ATO.CHTWTHHLD.000131
            Scheme for Trustee must be set to http://www.sbr.gov.au/id
    
            Legacy Rule Format:
            IF (Trustee:entity.identifier.scheme <> "http://www.sbr.gov.au/id")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.sbr.gov.au/id'
    
            Data Elements:
    
            ^IdentifierScheme = Trustee
            */
            assertion = ((report.TrusteeExists == true) && (report.TrusteeIdentifierScheme != @"http://www.sbr.gov.au/id"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CHTWTHHLD.000131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Trustee must be ""http://www.sbr.gov.au/id""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.TrusteeOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000131"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000131
    
            #region VR.ATO.CHTWTHHLD.000132
    
            /*  VR.ATO.CHTWTHHLD.000132
            Declaration Statement Accepted Indicator must be present for Lodgment
    
            Legacy Rule Format:
            IF (INT.TrueAndCorrect:pyin.xx.xx:Declaration.StatementAccepted.Indicator = NULL) AND (RP.TrueAndCorrect:pyin.xx.xx:Declaration.StatementAccepted.Indicator = NULL)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD90 = NULL) AND (^CHTWTHHLD78 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLD78 = RP.TrueAndCorrect:Declaration.StatementAccepted.Indicator
    
            ^CHTWTHHLD90 = INT.TrueAndCorrect:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.CHTWTHHLD90 == null && report.CHTWTHHLD78 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410078", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be present",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CHTWTHHLD.000132"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD90", Value = GetValueOrEmpty(report.CHTWTHHLD90) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD78", Value = GetValueOrEmpty(report.CHTWTHHLD78) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CHTWTHHLD.000132
    
            #region VR.ATO.GEN.000209
    
            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000209
    
            #region VR.ATO.GEN.000406
    
            /*  VR.ATO.GEN.000406
            Address field contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD3 <> BLANK) AND (StartsWithSet(^CHTWTHHLD3,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^CHTWTHHLD3, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CHTWTHHLD3 = RP.Postal:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD3) != true && IsMatch(report.CHTWTHHLD3, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.CHTWTHHLD3, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD3", Value = report.CHTWTHHLD3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000406
    
            #region VR.ATO.GEN.000408
    
            /*  VR.ATO.GEN.000408
            Address line 2 field cannot contain 'care of' reference.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD4 <> BLANK) AND (StartsWithSet(^CHTWTHHLD4,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^CHTWTHHLD4, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CHTWTHHLD4 = RP.Postal:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD4) != true && (IsMatch(report.CHTWTHHLD4, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.CHTWTHHLD4, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD4", Value = report.CHTWTHHLD4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000408
    
            #region VR.ATO.GEN.000458
    
            /*  VR.ATO.GEN.000458
            The Trustee Suffix code is not a valid code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD13 <> BLANK) AND (NotInDomain(^CHTWTHHLD13, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^CHTWTHHLD13 = Trustee:PersonNameDetails.NameSuffix.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD13) != true && !(IsMatch(report.CHTWTHHLD13, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD13", Value = report.CHTWTHHLD13 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000458
    
            #region VR.ATO.GEN.000459
    
            /*  VR.ATO.GEN.000459
            The Individual title code is not a valid code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD11 <> BLANK) AND (NotInDomain(^CHTWTHHLD11, 'TITLE CODES'))
    
            Data Elements:
    
            ^CHTWTHHLD11 = Trustee:PersonNameDetails.Title.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD11) != true && !(IsMatch(report.CHTWTHHLD11, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459", Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD11", Value = report.CHTWTHHLD11 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000459
    
            #region VR.ATO.GEN.000469
    
            /*  VR.ATO.GEN.000469
            Title, First name, Other given names or Suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD12 = BLANK) AND (^CHTWTHHLD13 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD13 = Trustee:PersonNameDetails.NameSuffix.Text
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD12) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD13) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000469", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is provided without Family Name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000469"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD13", Value = report.CHTWTHHLD13 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000469
    
            #region VR.ATO.GEN.000470
    
            /*  VR.ATO.GEN.000470
            Title, First name, Other given names or Suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD11 <> BLANK) AND (^CHTWTHHLD12 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD11 = Trustee:PersonNameDetails.Title.Text
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD11) != true && string.IsNullOrWhiteSpace(report.CHTWTHHLD12) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000470", Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is provided without Family Name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000470"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD11", Value = report.CHTWTHHLD11 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000470
    
            #region VR.ATO.GEN.000471
    
            /*  VR.ATO.GEN.000471
            Title, First name, Other given names or Suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text  = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD12  = BLANK) AND (^CHTWTHHLD14 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD12) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD14) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000471", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name is provided without Family name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000471"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000471
    
            #region VR.ATO.GEN.000472
    
            /*  VR.ATO.GEN.000472
            Other given name/s present without first name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD14 = BLANK) AND (^CHTWTHHLD15 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD14) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD15) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000472", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name is required",
                    LongDescription = @"Other given name is provided without First name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000472"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000472
    
            #region VR.ATO.GEN.000475
    
            /*  VR.ATO.GEN.000475
            Title, First name, Other given names or Suffix is present without the family name.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = NULLORBLANK)
            AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text  <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD12 = BLANK) AND (^CHTWTHHLD15  <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD12) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD15) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000475", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name is provided without Family name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000475"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000475
    
            #region VR.ATO.GEN.410039
    
            /*  VR.ATO.GEN.410039
            Non-individual name field only contains characters other than alphas and numerics, at least one alpha or numeric must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD2  <> BLANK) AND (NotContainsSet(^CHTWTHHLD2, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^CHTWTHHLD2 = RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD2) != true && !(IsMatch(report.CHTWTHHLD2, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD2", Value = report.CHTWTHHLD2 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410039
    
            #region VR.ATO.GEN.410039
    
            /*  VR.ATO.GEN.410039
            Non-individual name field only contains characters other than alphas and numerics, at least one alpha or numeric must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD16  <> BLANK) AND (NotContainsSet(^CHTWTHHLD16, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD16) != true && !(IsMatch(report.CHTWTHHLD16, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410039
    
            #region VR.ATO.GEN.410040
    
            /*  VR.ATO.GEN.410040
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD12 <> BLANK) AND (NotContainsSet(^CHTWTHHLD12, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD12) != true && !(IsMatch(report.CHTWTHHLD12, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410040"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410040
    
            #region VR.ATO.GEN.410063
    
            /*  VR.ATO.GEN.410063
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD14 <> BLANK) AND (NotContainsSet(^CHTWTHHLD14, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD14) != true && !(IsMatch(report.CHTWTHHLD14, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410063
    
            #region VR.ATO.GEN.410128
    
            /*  VR.ATO.GEN.410128
            Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.StatementAccepted.Indicator <> NULLORBLANK) AND (pyin.xx.xx:Declaration.StatementAccepted.Indicator = FALSE)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD78 <> BLANK) AND (^CHTWTHHLD78 = FALSE)
    
            Data Elements:
    
            ^CHTWTHHLD78 = RP.TrueAndCorrect:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.CHTWTHHLD78 != null && report.CHTWTHHLD78 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410128", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410128"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD78", Value = GetValueOrEmpty(report.CHTWTHHLD78) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410128
    
            #region VR.ATO.GEN.410128
    
            /*  VR.ATO.GEN.410128
            Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.StatementAccepted.Indicator <> NULLORBLANK) AND (pyin.xx.xx:Declaration.StatementAccepted.Indicator = FALSE)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD90 <> BLANK) AND (^CHTWTHHLD90 = FALSE)
    
            Data Elements:
    
            ^CHTWTHHLD90 = INT.TrueAndCorrect:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.CHTWTHHLD90 != null && report.CHTWTHHLD90 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410128", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410128"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD90", Value = GetValueOrEmpty(report.CHTWTHHLD90) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410128
    
            #region VR.ATO.GEN.410131
    
            /*  VR.ATO.GEN.410131
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD15 <> BLANK) AND (NotContainsSet(^CHTWTHHLD15, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD15) != true && !(IsMatch(report.CHTWTHHLD15, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410131
    
            #region VR.ATO.GEN.410200
    
            /*  VR.ATO.GEN.410200
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD12, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD12, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410200
    
            #region VR.ATO.GEN.410201
    
            /*  VR.ATO.GEN.410201
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used i
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD12, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD12, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410201
    
            #region VR.ATO.GEN.410202
    
            /*  VR.ATO.GEN.410202
            Individual name field contains suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the fi
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD14, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD14, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000436", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410202"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410202
    
            #region VR.ATO.GEN.410203
    
            /*  VR.ATO.GEN.410203
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD14, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD14, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410203
    
            #region VR.ATO.GEN.410204
    
            /*  VR.ATO.GEN.410204
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used i
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD14, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD14, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410204
           
            //Validation Rule VR.ATO.GEN.410206 removed for CR2900
    
            #region VR.ATO.GEN.410207
    
            /*  VR.ATO.GEN.410207
            Individual name field contains suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the fi
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD15, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD15, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000448", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410207
    
            #region VR.ATO.GEN.410208
    
            /*  VR.ATO.GEN.410208
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD15, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD15, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410208
    
            #region VR.ATO.GEN.410209
    
            /*  VR.ATO.GEN.410209
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used i
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD15, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD15, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410209
    
            #region VR.ATO.GEN.428026
    
            /*  VR.ATO.GEN.428026
            Address line 1 contains 'Care of' reference and no street address in Address line 2. If 'Care of' reference is on Address line 1, it must be followed by a name then Address line 2 must contain the street address.
    
            Legacy Rule Format:
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line 2.Text = NULLORBLANK))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^CHTWTHHLD3, 'C/-')) AND (^CHTWTHHLD4 = BLANK))
    
            Data Elements:
    
            ^CHTWTHHLD3 = RP.Postal:AddressDetails.Line1.Text
    
            ^CHTWTHHLD4 = RP.Postal:AddressDetails.Line2.Text
            */
            assertion = ((report.CHTWTHHLD3 == null ? false : report.CHTWTHHLD3.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.CHTWTHHLD4) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428026", Severity = ProcessMessageSeverity.Error,
                    Description = @"C/- must be followed by a Name and the Address on line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428026"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD3", Value = report.CHTWTHHLD3 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD4", Value = report.CHTWTHHLD4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428026
    
            #region VR.ATO.GEN.428036
    
            /*  VR.ATO.GEN.428036
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD6 <> BLANK) AND (NotInSet(^CHTWTHHLD6, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^CHTWTHHLD6 = RP.Postal:AddressDetails.StateOrTerritory.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD6) != true && !(IsMatch(report.CHTWTHHLD6, @"^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428036", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid State code",
                    LongDescription = @"The State code in the Current address is not Valid (Valid States are ""QLD "",""NSW "",""VIC "",""SA "",""WA "",""NT "",""ACT "",""TAS)",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428036"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD6", Value = report.CHTWTHHLD6 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428036
    
            #region VR.ATO.GEN.428041
    
            /*  VR.ATO.GEN.428041
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD7 <> BLANK) AND (IsNumeric(^CHTWTHHLD7) = 'FALSE' OR AsNumeric(^CHTWTHHLD7) < 200 OR AsNumeric(^CHTWTHHLD7) > 9999)
    
            Data Elements:
    
            ^CHTWTHHLD7 = RP.Postal:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD7) != true && (IsNumeric(report.CHTWTHHLD7) == false || AsNumeric(report.CHTWTHHLD7) < 200 || AsNumeric(report.CHTWTHHLD7) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428041", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD7", Value = report.CHTWTHHLD7 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428041
    
            #region VR.ATO.GEN.428231
    
            /*  VR.ATO.GEN.428231
            Individual name field contains suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the fi
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CHTWTHHLD12, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD12, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000422", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428231"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428231
    
            #region VR.ATO.GEN.428247
    
            /*  VR.ATO.GEN.428247
            Your Tax Agent number has failed the Tax Agent registration algorithm check. Input the correct Tax Agent registration number.
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD76 <> BLANK) AND (FailsTANAlgorithm(^CHTWTHHLD76))
    
            Data Elements:
    
            ^CHTWTHHLD76 = RP.TrueAndCorrect:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD76) != true && FailsTANAlgorithm(report.CHTWTHHLD76));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD76", Value = report.CHTWTHHLD76 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428247
    
            #region VR.ATO.GEN.428247
    
            /*  VR.ATO.GEN.428247
            Your Tax Agent number has failed the Tax Agent registration algorithm check. Input the correct Tax Agent registration number.
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD88 <> BLANK) AND (FailsTANAlgorithm(^CHTWTHHLD88))
    
            Data Elements:
    
            ^CHTWTHHLD88 = INT.TrueAndCorrect:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD88) != true && FailsTANAlgorithm(report.CHTWTHHLD88));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD88", Value = report.CHTWTHHLD88 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428247
    
            #region VR.ATO.GEN.428254
    
            /*  VR.ATO.GEN.428254
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD3, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLD3 = RP.Postal:AddressDetails.Line1.Text
            */
            assertion = (report.CHTWTHHLD3 == null ? false : report.CHTWTHHLD3.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD3", Value = report.CHTWTHHLD3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428254
    
            #region VR.ATO.GEN.428255
    
            /*  VR.ATO.GEN.428255
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD4, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLD4 = RP.Postal:AddressDetails.Line2.Text
            */
            assertion = (report.CHTWTHHLD4 == null ? false : report.CHTWTHHLD4.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD4", Value = report.CHTWTHHLD4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428255
    
            #region VR.ATO.GEN.428256
    
            /*  VR.ATO.GEN.428256
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD5, 'UNKNOWN')
    
            Data Elements:
    
            ^CHTWTHHLD5 = RP.Postal:AddressDetails.LocalityName.Text
            */
            assertion = (report.CHTWTHHLD5 == null ? false : report.CHTWTHHLD5.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD5", Value = report.CHTWTHHLD5 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428256
    
            #region VR.ATO.GEN.428258
    
            /*  VR.ATO.GEN.428258
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD2, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD2 = RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CHTWTHHLD2 == null ? false : report.CHTWTHHLD2.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD2", Value = report.CHTWTHHLD2 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428258
    
            #region VR.ATO.GEN.428258
    
            /*  VR.ATO.GEN.428258
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD16, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CHTWTHHLD16 == null ? false : report.CHTWTHHLD16.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428258
    
            #region VR.ATO.GEN.428260
    
            /*  VR.ATO.GEN.428260
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLD2, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLD2 = RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD2, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD2", Value = report.CHTWTHHLD2 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428260
    
            #region VR.ATO.GEN.428260
    
            /*  VR.ATO.GEN.428260
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLD16, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLD16 = Trustee:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD16, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD16", Value = report.CHTWTHHLD16 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428260
    
            #region VR.ATO.GEN.428262
    
            /*  VR.ATO.GEN.428262
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD12, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.CHTWTHHLD12 == null ? false : report.CHTWTHHLD12.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428262
    
            #region VR.ATO.GEN.428263
    
            /*  VR.ATO.GEN.428263
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLD12, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLD12 = Trustee:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD12, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD12", Value = report.CHTWTHHLD12 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428263
    
            #region VR.ATO.GEN.428264
    
            /*  VR.ATO.GEN.428264
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD14, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = (report.CHTWTHHLD14 == null ? false : report.CHTWTHHLD14.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428264
    
            #region VR.ATO.GEN.428265
    
            /*  VR.ATO.GEN.428265
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLD14, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLD14 = Trustee:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD14, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD14", Value = report.CHTWTHHLD14 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428265
    
            #region VR.ATO.GEN.428266
    
            /*  VR.ATO.GEN.428266
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD15, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.CHTWTHHLD15 == null ? false : report.CHTWTHHLD15.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428266
    
            #region VR.ATO.GEN.428267
    
            /*  VR.ATO.GEN.428267
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","'’","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^CHTWTHHLD15, '"--","''","  "'))
    
            Data Elements:
    
            ^CHTWTHHLD15 = Trustee:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD15, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD15", Value = report.CHTWTHHLD15 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428267
    
            #region VR.ATO.GEN.434223
    
            /*  VR.ATO.GEN.434223
            The Australian Business Number (ABN) has failed the ABN algorithm check.
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifer <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifer) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD80 <> BLANK) and (FailsABNAlgorithm(^CHTWTHHLD80))
    
            Data Elements:
    
            ^CHTWTHHLD80 = RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD80) != true && FailsABNAlgorithm(report.CHTWTHHLD80));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD80", Value = report.CHTWTHHLD80 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.434223
    
            #region VR.ATO.GEN.434223
    
            /*  VR.ATO.GEN.434223
            The Australian Business Number (ABN) has failed the ABN algorithm check.
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifer <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifer) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD17 <> BLANK) and (FailsABNAlgorithm(^CHTWTHHLD17))
    
            Data Elements:
    
            ^CHTWTHHLD17 = Trustee:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD17) != true && FailsABNAlgorithm(report.CHTWTHHLD17));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier[@contextRef='" + report.TrusteeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD17", Value = report.CHTWTHHLD17 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.434223
    
            #region VR.ATO.GEN.438070
    
            /*  VR.ATO.GEN.438070
            If an Intermediary is not present, Reporting Party Declarer contact name must be provided.
    
            Legacy Rule Format:
            IF ((CONTEXT(INT.TrueAndCorrect)<>NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^Context <> NULL) AND (^CHTWTHHLD84 = BLANK))
    
            Data Elements:
    
            ^CHTWTHHLD84 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CHTWTHHLD84) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438070", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer contact name must be provided",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer contact name must be provided.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438070"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD84", Value = report.CHTWTHHLD84 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438070
    
            #region VR.ATO.GEN.438071
    
            /*  VR.ATO.GEN.438071
            If an Intermediary is not present, Reporting Party Declarer contact name must be provided.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)=NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context=NULL) AND (^CHTWTHHLD73 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD73 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && string.IsNullOrWhiteSpace(report.CHTWTHHLD73) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438071", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer contact name must be provided",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer contact name must be provided.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438071"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD73", Value = report.CHTWTHHLD73 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438071
    
            #region VR.ATO.GEN.438072
    
            /*  VR.ATO.GEN.438072
            If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) <> NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CHTWTHHLD87 = BLANK) AND (^CHTWTHHLD86 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD87 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
    
            ^CHTWTHHLD86 = INT.TrueAndCorrect:ElectronicContact.Telephone.Area.Code
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CHTWTHHLD87) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD86) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438072", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD87", Value = report.CHTWTHHLD87 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD86", Value = report.CHTWTHHLD86 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438072
    
            #region VR.ATO.GEN.438073
    
            /*  VR.ATO.GEN.438073
            If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) = NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context = NULL) AND (^CHTWTHHLD79 = BLANK) AND (^CHTWTHHLD75 <> BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD75 = RP.TrueAndCorrect:ElectronicContact.Telephone.Area.Code
    
            ^CHTWTHHLD79 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && string.IsNullOrWhiteSpace(report.CHTWTHHLD79) == true && string.IsNullOrWhiteSpace(report.CHTWTHHLD75) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438073", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438073"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD79", Value = report.CHTWTHHLD79 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD75", Value = report.CHTWTHHLD75 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438073
    
            #region VR.ATO.GEN.438074
    
            /*  VR.ATO.GEN.438074
            If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) <> NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number <> NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CHTWTHHLD87 <> NULL) AND (^CHTWTHHLD86 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD87 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
    
            ^CHTWTHHLD86 = INT.TrueAndCorrect:ElectronicContact.Telephone.Area.Code
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.CHTWTHHLD87 != null && string.IsNullOrWhiteSpace(report.CHTWTHHLD86) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438074", Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is present, Intermediary area code is required.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD87", Value = report.CHTWTHHLD87 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD86", Value = report.CHTWTHHLD86 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438074
    
            #region VR.ATO.GEN.438075
    
            /*  VR.ATO.GEN.438075
            If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) = NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number <> NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context = NULL) AND (^CHTWTHHLD79 <> NULL) AND (^CHTWTHHLD75 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD75 = RP.TrueAndCorrect:ElectronicContact.Telephone.Area.Code
    
            ^CHTWTHHLD79 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && report.CHTWTHHLD79 != null && string.IsNullOrWhiteSpace(report.CHTWTHHLD75) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438075", Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD79", Value = report.CHTWTHHLD79 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD75", Value = report.CHTWTHHLD75 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438075
    
            #region VR.ATO.GEN.438076
    
            /*  VR.ATO.GEN.438076
            If an Intermediary is present, Intermediary Declarer Telephone number must be provided.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)<>NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CHTWTHHLD87 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD87 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CHTWTHHLD87) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438076", Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact telephone details must be provided",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer Telephone number must be provided.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD87", Value = report.CHTWTHHLD87 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438076
    
            #region VR.ATO.GEN.438077
    
            /*  VR.ATO.GEN.438077
            If an Intermediary is not present, Reporting Party Declarer Telephone number must be provided.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)=NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context=NULL) AND (^CHTWTHHLD79 = BLANK)
    
            Data Elements:
    
            ^CHTWTHHLD79 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && string.IsNullOrWhiteSpace(report.CHTWTHHLD79) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438077", Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact telephone details must be provided",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer Telephone number must be provided.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438077"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD79", Value = report.CHTWTHHLD79 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438077
    
            #region VR.ATO.GEN.438078
    
            /*  VR.ATO.GEN.438078
            If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)<>NULL) AND (INT.TrueAndCorrect:pyin.xx.xx:Declaration.Signature.Date = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CHTWTHHLD89 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLD89 = INT.TrueAndCorrect:Declaration.Signature.Date
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.CHTWTHHLD89 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438078", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signature Date is required",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer Signature Date must be present.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD89", Value = GetValueOrEmpty(report.CHTWTHHLD89) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438078
    
            #region VR.ATO.GEN.438079
    
            /*  VR.ATO.GEN.438079
            If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)=NULL) AND (RP.TrueAndCorrect:pyin.xx.xx:Declaration.Signature.Date = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context=NULL) AND (^CHTWTHHLD77 = NULL)
    
            Data Elements:
    
            ^CHTWTHHLD77 = RP.TrueAndCorrect:Declaration.Signature.Date
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && report.CHTWTHHLD77 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438079", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signature Date is required",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438079"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD77", Value = GetValueOrEmpty(report.CHTWTHHLD77) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438079
    
            #region VR.ATO.GEN.500030
    
            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^StartDate1 = INT.TrueAndCorrect
            */
            assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectStartDate != null && report.INTTrueAndCorrectStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500030
    
            #region VR.ATO.GEN.500030
    
            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^StartDate1 = RP:RP.Postal
            */
            assertion = ((report.RPPostalExists == true) && (report.RPPostalStartDate != null && report.RPPostalStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500030
    
            #region VR.ATO.GEN.500030
    
            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^StartDate1 = RP:RP.TrueAndCorrect
            */
            assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectStartDate != null && report.RPTrueAndCorrectStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500030
    
            #region VR.ATO.GEN.500030
    
            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^StartDate1 = Trustee
            */
            assertion = ((report.TrusteeExists == true) && (report.TrusteeStartDate != null && report.TrusteeStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500030
    
            #region VR.ATO.GEN.500033
    
            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> RP:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = RP
    
            ^EndDate1 = INT.TrueAndCorrect
            */
            assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectEndDate != null && report.INTTrueAndCorrectEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500033
    
            #region VR.ATO.GEN.500033
    
            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> RP:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = RP
    
            ^EndDate1 = RP:RP.Postal
            */
            assertion = ((report.RPPostalExists == true) && (report.RPPostalEndDate != null && report.RPPostalEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500033
    
            #region VR.ATO.GEN.500033
    
            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> RP:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = RP
    
            ^EndDate1 = RP:RP.TrueAndCorrect
            */
            assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectEndDate != null && report.RPTrueAndCorrectEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500033
    
            #region VR.ATO.GEN.500033
    
            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.endDate WHERE CONTEXT(ALL) <> RP:period.endDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^EndDate1 <> NULL AND ^EndDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = RP
    
            ^EndDate1 = Trustee
            */
            assertion = ((report.TrusteeExists == true) && (report.TrusteeEndDate != null && report.TrusteeEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500033
    
            #region VR.ATO.GEN.500035
    
            /*  VR.ATO.GEN.500035
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = INT.TrueAndCorrect
            */
            assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectStartDate > DateTime.Now.Date));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date cannot be any day in the future.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500035"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500035
    
            #region VR.ATO.GEN.500035
    
            /*  VR.ATO.GEN.500035
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = RP
            */
            assertion = ((report.RPExists == true) && (report.RPStartDate > DateTime.Now.Date));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date cannot be any day in the future.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500035"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500035
    
            #region VR.ATO.GEN.500035
    
            /*  VR.ATO.GEN.500035
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = RP:RP.Postal
            */
            assertion = ((report.RPPostalExists == true) && (report.RPPostalStartDate > DateTime.Now.Date));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date cannot be any day in the future.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPostalOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500035"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500035
    
            #region VR.ATO.GEN.500035
    
            /*  VR.ATO.GEN.500035
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = RP:RP.TrueAndCorrect
            */
            assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectStartDate > DateTime.Now.Date));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date cannot be any day in the future.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500035"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500035
    
            #region VR.ATO.GEN.500035
    
            /*  VR.ATO.GEN.500035
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = Trustee
            */
            assertion = ((report.TrusteeExists == true) && (report.TrusteeStartDate > DateTime.Now.Date));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date cannot be any day in the future.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.TrusteeOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500035"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500035
    
            #region VR.ATO.GEN.500037
    
            /*  VR.ATO.GEN.500037
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD73, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLD73 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD73, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500037", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500037"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD73", Value = report.CHTWTHHLD73 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500037
    
            #region VR.ATO.GEN.500037
    
            /*  VR.ATO.GEN.500037
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CHTWTHHLD84, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^CHTWTHHLD84 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = IsMatch(report.CHTWTHHLD84, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500037", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500037"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD84", Value = report.CHTWTHHLD84 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500037
    
            #region VR.ATO.GEN.500044
    
            /*  VR.ATO.GEN.500044
            Invalid context. The number of RP.Postal contexts must equal 1.
    
            Legacy Rule Format:
            IF COUNT(RP.Postal) <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP:RP.Postal
            */
            assertion = (Count(report.RPPostalCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500044", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.Postal contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPostalLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500044"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500044
    
            #region VR.ATO.GEN.500045
    
            /*  VR.ATO.GEN.500045
            Invalid context. The number of Trustee contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(Trustee) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = Trustee
            */
            assertion = (Count(report.TrusteeCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500045", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Trustee contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.TrusteeLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500045"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500045
    
            #region VR.ATO.GEN.500046
    
            /*  VR.ATO.GEN.500046
            Invalid context. The number of RP.TrueAndCorrect contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.TrueAndCorrect) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TrueAndCorrect
            */
            assertion = (Count(report.RPTrueAndCorrectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500046", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.TrueAndCorrect contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500046"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500046
    
            #region VR.ATO.GEN.500047
    
            /*  VR.ATO.GEN.500047
            Invalid context. The number of Intermediary.TrueAndCorrect contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(INT.TrueAndCorrect) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (Count(report.INTTrueAndCorrectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500047", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Intermediary.TrueAndCorrect contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500047"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500047
    
            #region VR.ATO.GEN.500050
    
            /*  VR.ATO.GEN.500050
            Ensure at least one declaration is provided for Intermediary or Reporting Party
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) = NULL) AND (CONTEXT(RP.TrueAndCorrect) = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context = NULL) AND (^RpContext = NULL)
    
            Data Elements:
    
            ^Context = INT.TrueAndCorrect
    
            ^RpContext = RP:RP.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && report.RPTrueAndCorrectExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500050", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. At least one declaration is required from either Intermediary or Reporting Party.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500050"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500050
    
            #region VR.ATO.GEN.500051
    
            /*  VR.ATO.GEN.500051
            Scheme for Intermediary.TrueAndCorrect must be set to http://www.abr.gov.au/abn
    
            Legacy Rule Format:
            IF (INT.TrueAndCorrect:entity.identifier.scheme <> “http://www.abr.gov.au/abn”)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.abr.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = INT.TrueAndCorrect
            */
            assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectIdentifierScheme != @"http://www.abr.gov.au/abn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500051", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Intermediary.TrueAndCorrect must be ""http://www.abr.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500051"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500051
    
            #region VR.ATO.GEN.500053
    
            /*  VR.ATO.GEN.500053
            Scheme for RP.TrueAndCorrect must be set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF (RP.TrueAndCorrect:entity.identifier.scheme <> “http://www.ato.gov.au/tfn”)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TrueAndCorrect
            */
            assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty.TrueAndCorrect must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500053"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500053
    
            #region VR.ATO.GEN.500054
    
            /*  VR.ATO.GEN.500054
            ABN in the context definition must pass the ABN algorithm check
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)<>NULL) AND (ABNALGORITHM(INT.TrueAndCorrect:entity.identifier.ABN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (FailsABNAlgorithm(^ABN))
    
            Data Elements:
    
            ^Context = INT.TrueAndCorrect
    
            ^ABN = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && FailsABNAlgorithm(report.INTTrueAndCorrectIdentifierABN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary declarer Australian Business Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500054"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500054
    
            #region VR.ATO.GEN.500055
    
            /*  VR.ATO.GEN.500055
            Declarer Contact Name has a maximum field length of 25
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 25
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD73) > 25
    
            Data Elements:
    
            ^CHTWTHHLD73 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (Length(report.CHTWTHHLD73) > 25);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name must not exceed 200 characters",
                    LongDescription = @"For Declarer Contact Name, the maximum allowable characters for this form is 200",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD73", Value = report.CHTWTHHLD73 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500055
    
            #region VR.ATO.GEN.500055
    
            /*  VR.ATO.GEN.500055
            Declarer Contact Name has a maximum field length of 25
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 25
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD84) > 25
    
            Data Elements:
    
            ^CHTWTHHLD84 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (Length(report.CHTWTHHLD84) > 25);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name must not exceed 200 characters",
                    LongDescription = @"For Declarer Contact Name, the maximum allowable characters for this form is 200",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD84", Value = report.CHTWTHHLD84 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500055
    
            #region VR.ATO.GEN.500057
    
            /*  VR.ATO.GEN.500057
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD73, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD73 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.CHTWTHHLD73 == null ? false : report.CHTWTHHLD73.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD73", Value = report.CHTWTHHLD73 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500057
    
            #region VR.ATO.GEN.500057
    
            /*  VR.ATO.GEN.500057
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CHTWTHHLD84, ' - ')
    
            Data Elements:
    
            ^CHTWTHHLD84 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.CHTWTHHLD84 == null ? false : report.CHTWTHHLD84.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD84", Value = report.CHTWTHHLD84 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500057
    
            #region VR.ATO.GEN.500058
    
            /*  VR.ATO.GEN.500058
            Declarer Position held of person making the declaration has a maximum field length of 50
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.Position.Text) > 50
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD74) > 50
    
            Data Elements:
    
            ^CHTWTHHLD74 = RP.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (Length(report.CHTWTHHLD74) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500058", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position must not exceed 50 characters",
                    LongDescription = @"For Declarer position, the maximum allowable characters for this form is 50",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD74", Value = report.CHTWTHHLD74 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500058
    
            #region VR.ATO.GEN.500058
    
            /*  VR.ATO.GEN.500058
            Declarer Position held of person making the declaration has a maximum field length of 50
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.Position.Text) > 50
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CHTWTHHLD85) > 50
    
            Data Elements:
    
            ^CHTWTHHLD85 = INT.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (Length(report.CHTWTHHLD85) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500058", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position must not exceed 50 characters",
                    LongDescription = @"For Declarer position, the maximum allowable characters for this form is 50",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD85", Value = report.CHTWTHHLD85 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500058
    
            #region VR.ATO.GEN.500059
    
            /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Position.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:PersonNameDetails.Position.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD74 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD74, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD74 = RP.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD74) != true && !(IsMatch(report.CHTWTHHLD74, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500059", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position held of person making the declaration contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD74", Value = report.CHTWTHHLD74 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500059
    
            #region VR.ATO.GEN.500059
    
            /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Position.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:PersonNameDetails.Position.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD85 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD85, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD85 = INT.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD85) != true && !(IsMatch(report.CHTWTHHLD85, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500059", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position held of person making the declaration contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD85", Value = report.CHTWTHHLD85 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500059
    
            #region VR.ATO.GEN.500065
    
            /*  VR.ATO.GEN.500065
            Tax agent number is not provided for reporting party as only applicable for Tax agent (Intermediary)
    
            Legacy Rule Format:
            IF (CONTEXT(RP.TrueAndCorrect) <> NULL) AND (RP.TrueAndCorrect:pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CHTWTHHLD76 <> NULL)
    
            Data Elements:
    
            ^CHTWTHHLD76 = RP.TrueAndCorrect:Identifiers.TaxAgentNumber.Identifier
    
            ^Context = RP:RP.TrueAndCorrect
            */
            assertion = (report.RPTrueAndCorrectExists != false && report.CHTWTHHLD76 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500065", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer tax agent number must not be present for Reporting Party",
                    LongDescription = @"Declarer tax agent number is only applicable for an Intermediary tax agent",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500065"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD76", Value = report.CHTWTHHLD76 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500065
    
            #region VR.ATO.GEN.500066
    
            /*  VR.ATO.GEN.500066
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD77 > Today())
    
            Data Elements:
    
            ^CHTWTHHLD77 = RP.TrueAndCorrect:Declaration.Signature.Date
            */
            assertion = (report.CHTWTHHLD77.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD77", Value = GetValueOrEmpty(report.CHTWTHHLD77) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500066
    
            #region VR.ATO.GEN.500066
    
            /*  VR.ATO.GEN.500066
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD89 > Today())
    
            Data Elements:
    
            ^CHTWTHHLD89 = INT.TrueAndCorrect:Declaration.Signature.Date
            */
            assertion = (report.CHTWTHHLD89.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD89", Value = GetValueOrEmpty(report.CHTWTHHLD89) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500066
    
            #region VR.ATO.GEN.500069
    
            /*  VR.ATO.GEN.500069
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD81 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD81, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD81 = RP.TrueAndCorrect:Declaration.SignatoryIdentifier.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD81) != true && !(IsMatch(report.CHTWTHHLD81, @"^[a-zA-Z0-9!@\$%&\*\(\)\-_=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500069", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer identifier contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD81", Value = report.CHTWTHHLD81 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500069
    
            #region VR.ATO.GEN.500069
    
            /*  VR.ATO.GEN.500069
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CHTWTHHLD91 <> BLANK) AND (NotCharacterInSet(^CHTWTHHLD91, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^CHTWTHHLD91 = INT.TrueAndCorrect:Declaration.SignatoryIdentifier.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CHTWTHHLD91) != true && !(IsMatch(report.CHTWTHHLD91, @"^[a-zA-Z0-9!@\$%&\*\(\)\-_=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500069", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer identifier contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CHTWTHHLD91", Value = report.CHTWTHHLD91 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500069
    
            #region VR.ATO.GEN.500075
    
            /*  VR.ATO.GEN.500075
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RP:entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
    
            ^IdentifierScheme = RP
            */
            assertion = ((report.RPExists == true) && (report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500075"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500075

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 